import * as zod from 'zod';
import { joinPath } from './utils/joinPath';
export class ValidationError extends Error {
    details;
    name;
    constructor(message, details = []) {
        super(message);
        this.details = details;
        this.name = 'ZodValidationError';
    }
    toString() {
        return this.message;
    }
}
function fromZodIssue(issue, issueSeparator, unionSeparator) {
    if (issue.code === 'invalid_union') {
        return issue.unionErrors
            .reduce((acc, zodError) => {
            const newIssues = zodError.issues
                .map((issue) => fromZodIssue(issue, issueSeparator, unionSeparator))
                .join(issueSeparator);
            if (!acc.includes(newIssues)) {
                acc.push(newIssues);
            }
            return acc;
        }, [])
            .join(unionSeparator);
    }
    if (issue.path.length > 0) {
        return `${issue.message} at "${joinPath(issue.path)}"`;
    }
    return issue.message;
}
export function fromZodError(zodError, options = {}) {
    const { maxIssuesInMessage = 99, issueSeparator = '; ', unionSeparator = ', or ', prefixSeparator = ': ', prefix = 'Validation error', } = options;
    const reason = zodError.errors
        .slice(0, maxIssuesInMessage)
        .map((issue) => fromZodIssue(issue, issueSeparator, unionSeparator))
        .join(issueSeparator);
    const message = reason ? [prefix, reason].join(prefixSeparator) : prefix;
    return new ValidationError(message, zodError.errors);
}
export const toValidationError = (options = {}) => (err) => {
    if (err instanceof zod.ZodError) {
        return fromZodError(err, options);
    }
    if (err instanceof Error) {
        return err;
    }
    return new Error('Unknown error');
};
export function isValidationError(err) {
    return err instanceof ValidationError;
}
export function isValidationErrorLike(err) {
    return err instanceof Error && err.name === 'ZodValidationError';
}
