"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Transaction = exports.OPERATION_TYPE = exports.ISOLATION_LEVEL = void 0;
exports.assertValidIsolationLevel = assertValidIsolationLevel;
exports.isolationLevelByValue = void 0;

var _writableTrackingBuffer = _interopRequireDefault(require("./tracking-buffer/writable-tracking-buffer"));

var _allHeaders = require("./all-headers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
  s2.2.6.8
 */
const OPERATION_TYPE = {
  TM_GET_DTC_ADDRESS: 0x00,
  TM_PROPAGATE_XACT: 0x01,
  TM_BEGIN_XACT: 0x05,
  TM_PROMOTE_XACT: 0x06,
  TM_COMMIT_XACT: 0x07,
  TM_ROLLBACK_XACT: 0x08,
  TM_SAVE_XACT: 0x09
};
exports.OPERATION_TYPE = OPERATION_TYPE;
const ISOLATION_LEVEL = {
  NO_CHANGE: 0x00,
  READ_UNCOMMITTED: 0x01,
  READ_COMMITTED: 0x02,
  REPEATABLE_READ: 0x03,
  SERIALIZABLE: 0x04,
  SNAPSHOT: 0x05
};
exports.ISOLATION_LEVEL = ISOLATION_LEVEL;
const isolationLevelByValue = {};
exports.isolationLevelByValue = isolationLevelByValue;

for (const name in ISOLATION_LEVEL) {
  const value = ISOLATION_LEVEL[name];
  isolationLevelByValue[value] = name;
}

function assertValidIsolationLevel(isolationLevel, name) {
  if (typeof isolationLevel !== 'number') {
    throw new TypeError(`The "${name}" ${name.includes('.') ? 'property' : 'argument'} must be of type number. Received type ${typeof isolationLevel} (${isolationLevel})`);
  }

  if (!Number.isInteger(isolationLevel)) {
    throw new RangeError(`The value of "${name}" is out of range. It must be an integer. Received: ${isolationLevel}`);
  }

  if (!(isolationLevel >= 0 && isolationLevel <= 5)) {
    throw new RangeError(`The value of "${name}" is out of range. It must be >= 0 && <= 5. Received: ${isolationLevel}`);
  }
}

class Transaction {
  constructor(name, isolationLevel = ISOLATION_LEVEL.NO_CHANGE) {
    this.name = void 0;
    this.isolationLevel = void 0;
    this.outstandingRequestCount = void 0;
    this.name = name;
    this.isolationLevel = isolationLevel;
    this.outstandingRequestCount = 1;
  }

  beginPayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ucs2');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_BEGIN_XACT);
    buffer.writeUInt8(this.isolationLevel);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2');
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },

      toString: () => {
        return 'Begin Transaction: name=' + this.name + ', isolationLevel=' + isolationLevelByValue[this.isolationLevel];
      }
    };
  }

  commitPayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ascii');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_COMMIT_XACT);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2'); // No fBeginXact flag, so no new transaction is started.

    buffer.writeUInt8(0);
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },

      toString: () => {
        return 'Commit Transaction: name=' + this.name;
      }
    };
  }

  rollbackPayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ascii');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_ROLLBACK_XACT);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2'); // No fBeginXact flag, so no new transaction is started.

    buffer.writeUInt8(0);
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },

      toString: () => {
        return 'Rollback Transaction: name=' + this.name;
      }
    };
  }

  savePayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ascii');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_SAVE_XACT);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2');
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },

      toString: () => {
        return 'Save Transaction: name=' + this.name;
      }
    };
  }

  isolationLevelToTSQL() {
    switch (this.isolationLevel) {
      case ISOLATION_LEVEL.READ_UNCOMMITTED:
        return 'READ UNCOMMITTED';

      case ISOLATION_LEVEL.READ_COMMITTED:
        return 'READ COMMITTED';

      case ISOLATION_LEVEL.REPEATABLE_READ:
        return 'REPEATABLE READ';

      case ISOLATION_LEVEL.SERIALIZABLE:
        return 'SERIALIZABLE';

      case ISOLATION_LEVEL.SNAPSHOT:
        return 'SNAPSHOT';
    }

    return '';
  }

}

exports.Transaction = Transaction;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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