"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
const MAX = (1 << 16) - 1;
const UNKNOWN_PLP_LEN = Buffer.from([0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff]);
const PLP_TERMINATOR = Buffer.from([0x00, 0x00, 0x00, 0x00]);
const NULL_LENGTH = Buffer.from([0xFF, 0xFF]);
const MAX_NULL_LENGTH = Buffer.from([0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF]);
const VarBinary = {
  id: 0xA5,
  type: 'BIGVARBIN',
  name: 'VarBinary',
  maximumLength: 8000,
  declaration: function (parameter) {
    const value = parameter.value; // Temporary solution. Remove 'any' later.

    let length;

    if (parameter.length) {
      length = parameter.length;
    } else if (value != null) {
      length = value.length || 1;
    } else if (value === null && !parameter.output) {
      length = 1;
    } else {
      length = this.maximumLength;
    }

    if (length <= this.maximumLength) {
      return 'varbinary(' + length + ')';
    } else {
      return 'varbinary(max)';
    }
  },
  resolveLength: function (parameter) {
    const value = parameter.value; // Temporary solution. Remove 'any' later.

    if (parameter.length != null) {
      return parameter.length;
    } else if (value != null) {
      return value.length;
    } else {
      return this.maximumLength;
    }
  },
  generateTypeInfo: function (parameter) {
    const buffer = Buffer.alloc(3);
    buffer.writeUInt8(this.id, 0);

    if (parameter.length <= this.maximumLength) {
      buffer.writeUInt16LE(parameter.length, 1);
    } else {
      buffer.writeUInt16LE(MAX, 1);
    }

    return buffer;
  },

  generateParameterLength(parameter, options) {
    if (parameter.value == null) {
      if (parameter.length <= this.maximumLength) {
        return NULL_LENGTH;
      } else {
        return MAX_NULL_LENGTH;
      }
    }

    let value = parameter.value;

    if (!Buffer.isBuffer(value)) {
      value = value.toString();
    }

    const length = Buffer.byteLength(value, 'ucs2');

    if (parameter.length <= this.maximumLength) {
      const buffer = Buffer.alloc(2);
      buffer.writeUInt16LE(length, 0);
      return buffer;
    } else {
      // writePLPBody
      return UNKNOWN_PLP_LEN;
    }
  },

  *generateParameterData(parameter, options) {
    if (parameter.value == null) {
      return;
    }

    let value = parameter.value;

    if (parameter.length <= this.maximumLength) {
      if (Buffer.isBuffer(value)) {
        yield value;
      } else {
        yield Buffer.from(value.toString(), 'ucs2');
      }
    } else {
      // writePLPBody
      if (!Buffer.isBuffer(value)) {
        value = value.toString();
      }

      const length = Buffer.byteLength(value, 'ucs2');

      if (length > 0) {
        const buffer = Buffer.alloc(4);
        buffer.writeUInt32LE(length, 0);
        yield buffer;

        if (Buffer.isBuffer(value)) {
          yield value;
        } else {
          yield Buffer.from(value, 'ucs2');
        }
      }

      yield PLP_TERMINATOR;
    }
  },

  validate: function (value) {
    if (value == null) {
      return null;
    }

    if (!Buffer.isBuffer(value)) {
      throw new TypeError('Invalid buffer.');
    }

    return value;
  }
};
var _default = VarBinary;
exports.default = _default;
module.exports = VarBinary;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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