"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.keySize = exports.generateKeySalt = exports.deriveKey = exports.AeadAes256CbcHmac256EncryptionKey = void 0;

var _crypto = require("crypto");

var _symmetricKey = _interopRequireDefault(require("./symmetric-key"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// This code is based on the `mssql-jdbc` library published under the conditions of MIT license.
// Copyright (c) 2019 Microsoft Corporation
const keySize = 256;
exports.keySize = keySize;
const keySizeInBytes = keySize / 8;

const deriveKey = (rootKey, salt) => {
  const hmac = (0, _crypto.createHmac)('sha256', rootKey);
  hmac.update(Buffer.from(salt, 'utf16le'));
  return hmac.digest();
};

exports.deriveKey = deriveKey;

const generateKeySalt = (keyType, algorithmName, keySize) => `Microsoft SQL Server cell ${keyType} key ` + `with encryption algorithm:${algorithmName} and key length:${keySize}`;

exports.generateKeySalt = generateKeySalt;

class AeadAes256CbcHmac256EncryptionKey extends _symmetricKey.default {
  constructor(rootKey, algorithmName) {
    super(rootKey);
    this.algorithmName = void 0;
    this.encryptionKeySaltFormat = void 0;
    this.macKeySaltFormat = void 0;
    this.ivKeySaltFormat = void 0;
    this.encryptionKey = void 0;
    this.macKey = void 0;
    this.ivKey = void 0;
    this.algorithmName = algorithmName;
    this.encryptionKeySaltFormat = generateKeySalt('encryption', this.algorithmName, keySize);
    this.macKeySaltFormat = generateKeySalt('MAC', this.algorithmName, keySize);
    this.ivKeySaltFormat = generateKeySalt('IV', this.algorithmName, keySize);

    if (rootKey.length !== keySizeInBytes) {
      throw new Error(`The column encryption key has been successfully decrypted but it's length: ${rootKey.length} does not match the length: ${keySizeInBytes} for algorithm "${this.algorithmName}". Verify the encrypted value of the column encryption key in the database.`);
    }

    try {
      const encKeyBuff = deriveKey(rootKey, this.encryptionKeySaltFormat);
      this.encryptionKey = new _symmetricKey.default(encKeyBuff);
      const macKeyBuff = deriveKey(rootKey, this.macKeySaltFormat);
      this.macKey = new _symmetricKey.default(macKeyBuff);
      const ivKeyBuff = deriveKey(rootKey, this.ivKeySaltFormat);
      this.ivKey = new _symmetricKey.default(ivKeyBuff);
    } catch (error) {
      throw new Error(`Key extraction failed : ${error.message}.`);
    }
  }

  getEncryptionKey() {
    return this.encryptionKey.rootKey;
  }

  getMacKey() {
    return this.macKey.rootKey;
  }

  getIvKey() {
    return this.ivKey.rootKey;
  }

}

exports.AeadAes256CbcHmac256EncryptionKey = AeadAes256CbcHmac256EncryptionKey;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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